/* UNIVERSAL - CalculiX interface program,  Bernhardi, Aug 2011   */
/****************************************************************************
* THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
* IT UNDER THE TERMS OF THE GNU LESSER GENERAL PUBLIC LICENSE 
* AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
*  
* THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
* WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
* MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
* LESSER GENERAL PUBLIC LICENSE FOR MORE DETAILS.
*
* YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
* ALONG WITH THIS PROGRAM; IF NOT, WRITE TO THE FREE SOFTWARE FOUNDATION, INC,
* 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*****************************************************************************/
/*
This small program converts finite elemet meshes given in the SDRL universal
file format into ABAQUS / Calculix compatible input decks. The format 
of the universal data sets can be found at http://www.sdrl.uc.edu.

The element - node mapping is taken from the Python unv2abaqus script, which 
can be found at /www.caelinux.org.

This program will still be far from being error - free; therefore, use 
it with caution!

The program also translates surface meshes in case they are present in
the .med file: If the main mesh is planar, the line elements (if they
exist) are translated to abaqus elsets with load surface IDs P1...P4.

Correspondingly, if a volume mesh is given, the surface meshes (if they
are also present in the med file) are converted to elsets with load
IDs P1...P6.

The *elset names will be the same as the group names defined in SALOME.

I use that feature to introduce boundary conditions. However, the
calculation of the surface elsets takes quite long for very large meshes. 

The program works for all CalculiX element types. It compiles under
opensuse 12.3 (but should compile on all linuxes). 

To install and use: 

- place the program source text into a proper directory, e.g. /path/unical

- go to that directory.

- issue a "gcc -o2 -static -o unical unical2.c" command.

- try it out using, for example, a universal file cube.unv: 
"/path/unical cube".

- Then, configure your Abaqus/Calculix analysis run by modifying the
cube.inp file: Introduction of loads, boundary conditions, prodedures, 
etc.

Remarks: 

- Unical translates the following three Universal data sets: 
  2411, node point numbers and coordinates
  2412, element type codes and connectivities
  2477, groups of nodes and elements.
Everything else is ignored. 

- Only tested with universal files generated by SALOME.

- Planar elements are always translated to shell elements, e.g. S8R. Thus
the *element, type= lines must be modified if e.g. CPS8 elements are to be 
used. 

- In case of a volume mesh, SALOME also generates surface elements; 
correspondingly, in case of a planar mesh the boundary is also output 
using line elements. In most cases the boundaries are not needed and 
are therefore not translated to CalculiX; rather, the surface elements
are used to generate *surface input lines for ease of boundary condition 
definition, as described above. 

- However, there are cases where different element dimensions are desired; 
for example, combination of volume elements and beam elements. If this is the 
case, specify "./unical -f cube". Then, all elements are translated. If only 
volumes and beams are wanted, shell elements must be manually removed from 
the .inp file. 

- Beam orientations are not treated yet.
=========================================================================*/

#define PROOL 1 // modification by prool: alternative output format.
// Prool here: http://prool.kharkov.org http://calculix.kharkov.org <proolix@gmail.com>

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h> // add by prool

void exitif();             /* return error message and exit. */
void countuni();           /* count number of nodes and elements. */
void allocate();           /* allocate large data arrays. */
void readnodes();          /* read node point data from .unv file. */
void readelements();       /* read element connectivities from .unv file. */
void readgroups();         /* read group information. */
void writemesh();          /* write mesh data to med file. */
void surface();            /* generate and write surfaces. */

FILE *fid=0;               /* UNIVERSAL mesh file. */
char inname[200];          /* input file name: .unv file. */
char outname[200];         /* output file name: .inp file. */
char problem[200];         /* problem name. */
int full;                  /* full.eq.0; only elements with dimension less than */
                           /* max dimension are converted; full.eq.1, all elements. */
int ret;                   /* return key. */
char *sret;                /* return key for fgets(). */
char line[201];            /* this holds one input line. */
char name[101];            /* name of dataset. */
int maxline=200;           /* max 100 characters in one input line. */
int key;                   /* .unv file record key. */
int node;                  /* node counter. */
int element;               /* element counter. */
int maxnode;               /* total number of nodes. */
int maxele[16];            /* total number of elements for each type. */
int sumele;                /* total number of all elements. */
int numnode;               /* current number of node. */
int numvol;                /* number of elements with dim=maxdimen. */
int numele;                /* current number of element. */
int elnumber[16];          /* number of elements for each type. */
int *nodenumbers;          /* nope point numbers; sorted. */
double *coords;            /* node point coordinates. */
int *elnumbers[16];        /* element numbers. */
int *elem[16];             /* element connectivity. */
double x,y,z;              /* node coordinates. */
int n[101];                /* nodes at current element. */
int eltype;                /* type of current element (internal). */
int utype;                 /* type of element, universal file. */
int btype;                 /* beam or non-beam element */
int dummy;                 /* nothing. */
int elnode[50];            /* numbers of nodes at current element. */
int nc,nc1;                /* counts nodes during reading of elements. */
int i,j,k;                 /* loop counters. */
char groupname[101];       /* group name. */
int groupnumber;           /* group number. */
int nument;                /* number of entities in current group. */
int maxgroup;              /* total number of groups on dataset. */
int numgroup;              /* current group number. */
int *groupents;            /* lengths of all groups. */
int *groupnumbers;         /* stores all group numbers. */
char **groupnames;         /* stores all group names. */
int **groupmembers;        /* stores all group members (elem/node numbers). */
int **grouptypes;          /* types of group members (1=elem, 2=node). */
int **grouppointers;       /* pointers to connectivity of elements in groups. */
int **groupeltypes;        /* types of elements in groups. */

/* a plane element listed in surfnum corresponds to 
a volume element listed in volelnum and a surface number 
(1-6, listed in facenum) of that volume element. */
int *surfnum;              /* surface element number. */
int *volelnum;             /* volume element number. */
int *facenum;              /* face number of volume element. */
int *group;                /* group number of surface element. */

int eltypes[87][3]={
 0,  0, 0,                 /* empty. */
11, 11, 1,                 /* Rod  */
11, 21, 1,                 /* Linear beam */
12, 22, 1,                 /* Tapered beam */
 0, 23, 1,                 /* Curved beam */
12, 24, 1,                 /* Parabolic beam */
 0, 31, 1,                 /* Straight pipe */
 0, 32, 1,                 /* Curved pipe */
 7, 41, 0,                 /* Plane Stress Linear Triangle */
 8, 42, 0,                 /* Plane Stress Parabolic Triangle */
 0, 43, 0,                 /* Plane Stress Cubic Triangle */
 9, 44, 0,                 /* Plane Stress Linear Quadrilateral */
10, 45, 0,                 /* Plane Stress Parabolic Quadrilateral */
 0, 46, 0,                 /* Plane Strain Cubic Quadrilateral */
 7, 51, 0,                 /* Plane Strain Linear Triangle */
 8, 52, 0,                 /* Plane Strain Parabolic Triangle */
 0, 53, 0,                 /* Plane Strain Cubic Triangle */
 9, 54, 0,                 /* Plane Strain Linear Quadrilateral */
10, 55, 0,                 /* Plane Strain Parabolic Quadrilateral */
 0, 56, 0,                 /* Plane Strain Cubic Quadrilateral */
 7, 61, 0,                 /* Plate Linear Triangle */
 8, 62, 0,                 /* Plate Parabolic Triangle */
 0, 63, 0,                 /* Plate Cubic Triangle */
 9, 64, 0,                 /* Plate Linear Quadrilateral */
10, 65, 0,                 /* Plate Parabolic Quadrilateral */
 0, 66, 0,                 /* Plate Cubic Quadrilateral */
 9, 71, 0,                 /* Membrane Linear Quadrilateral */
 8, 72, 0,                 /* Membrane Parabolic Triangle */
10, 73, 0,                 /* Membrane Cubic Triangle */
 7, 74, 0,                 /* Membrane Linear Triangle */
 0, 75, 0,                 /* Membrane Parabolic Quadrilateral */
 0, 76, 0,                 /* Membrane Cubic Quadrilateral */
 7, 81, 0,                 /* Axisymetric Solid Linear Triangle */
 8, 82, 0,                 /* Axisymetric Solid Parabolic Triangle */
 9, 84, 0,                 /* Axisymetric Solid Linear Quadrilateral */
10, 85, 0,                 /* Axisymetric Solid Parabolic Quadrilateral */
 7, 91, 0,                 /* Thin Shell Linear Triangle */
 8, 92, 0,                 /* Thin Shell Parabolic Triangle */
 0, 93, 0,                 /* Thin Shell Cubic Triangle */
 9, 94, 0,                 /* Thin Shell Linear Quadrilateral */
10, 95, 0,                 /* Thin Shell Parabolic Quadrilateral */
 0, 96, 0,                 /* Thin Shell Cubic Quadrilateral */
 0,101, 0,                 /* Thick Shell Linear Wedge */
 0,102, 0,                 /* Thick Shell Parabolic Wedge */
 0,103, 0,                 /* Thick Shell Cubic Wedge */
 0,104, 0,                 /* Thick Shell Linear Brick */
 0,105, 0,                 /* Thick Shell Parabolic Brick */
 0,106, 0,                 /* Thick Shell Cubic Brick */
 3,111, 0,                 /* Solid Linear Tetrahedron */
 2,112, 0,                 /* Solid Linear Wedge */
 5,113, 0,                 /* Solid Parabolic Wedge */
 0,114, 0,                 /* Solid Cubic Wedge */
 1,115, 0,                 /* Solid Linear Brick */
 4,116, 0,                 /* Solid Parabolic Brick */
 0,117, 0,                 /* Solid Cubic Brick */
 6,118, 0,                 /* Solid Parabolic Tetrahedron */
 0,121, 0,                 /* Rigid Bar */
 0,122, 0,                 /* Rigid Element */
 0,136, 0,                 /* Node To Node Translational Spring */
 0,137, 0,                 /* Node To Node Rotational Spring */
 0,138, 0,                 /* Node To Ground Translational Spring */
 0,139, 0,                 /* Node To Ground Rotational Spring */
 0,141, 0,                 /* Node To Node Damper */
 0,142, 0,                 /* Node To Gound Damper */
 0,151, 0,                 /* Node To Node Gap */
 0,152, 0,                 /* Node To Ground Gap */
 0,161, 0,                 /* Lumped Mass */
 0,171, 0,                 /* Axisymetric Linear Shell */
 0,172, 0,                 /* Axisymetric Parabolic Shell */
 0,181, 0,                 /* Constraint */
 0,191, 0,                 /* Plastic Cold Runner */
 0,192, 0,                 /* Plastic Hot Runner */
 0,193, 0,                 /* Plastic Water Line */
 0,194, 0,                 /* Plastic Fountain */
 0,195, 0,                 /* Plastic Baffle */
 0,196, 0,                 /* Plastic Rod Heater */
 0,201, 0,                 /* Linear node-to-node interface */
 0,202, 0,                 /* Linear edge-to-edge interface */
 0,203, 0,                 /* Parabolic edge-to-edge interface */
 0,204, 0,                 /* Linear face-to-face interface */
 0,208, 0,                 /* Parabolic face-to-face interface */
 0,212, 0,                 /* Linear axisymmetric interface */
 0,213, 0,                 /* Parabolic axisymmetric interface */
 0,221, 0,                 /* Linear rigid surface */
 0,222, 0,                 /* Parabolic rigin surface */
 0,231, 0,                 /* Axisymetric linear rigid surface */
 0,232, 0,                 /* Axisymetric parabolic rigid surface */  };

/* eltype=          0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12, 0, 0, 0   */
int numnodes[16]=  {0, 8, 6, 4,20,15,10, 3, 6, 4, 8, 2, 3, 0, 0, 0};
int smatch[2][16]= {0, 9, 7, 7,10,10, 8,11,12,11,12, 0, 0, 0, 0, 0,
                    0, 0, 9, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int dimensions[16]={0, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 1, 1, 0, 0, 0};

char nomabq[16][10]=                                                    {
"null", "C3D8", "C3D6", "C3D4", "C3D20", "C3D15", "C3D10", "S3",
"S6", "S4R", "S8R", "B31", "B32", "null", "null", "null",               };

int mapunv[16][21]=                                                     {
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* null */
 1, 1, 2, 3, 4, 5, 6, 7, 8, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* C3D8 */
 2, 1, 2, 3, 4, 5, 6, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* C3D6 */
 3, 1, 2, 3, 4, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* C3D4 */
 4, 1, 3, 5, 7,13,15,17,19, 2, 4,  6, 8,14,16,18,20, 9,10,11,12, /* C3D20*/
 5, 1, 3, 5,10,12,14, 2, 4, 6,11, 13,15, 7, 8, 9, 0, 0, 0, 0, 0, /* C3D15*/
 6, 1, 3, 5,10, 2, 4, 6, 7, 8, 9,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* C3D10*/
 7, 1, 3, 2, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* S3   */
 8, 1, 5, 3, 6, 4, 2, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* S6   */
 9, 1, 4, 3, 2, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* S4R  */
10, 1, 7, 5, 3, 8, 6, 4, 2, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* S8R  */
11, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* B31  */
12, 1, 2, 3, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* B32  */
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* null */
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* null */
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* null */};

/* pointers to faces tables. */
int elpointer[15]={0,2,1,3,2,1,3,5,5,4,4,0,0,0,0};

/* faces contains ABAQUS convention of node numbers at element faces. */
/* first line is empty; other 6 lines contain node numbers at max. 6 faces. */
/* faces[i][0][0]  (upperleft entries): Number of faces at this element. */
/* faces[i][j][0]  (left columns): Number of nodes at faces. */
int  faces[6][7][5]   =                                                 {

0, 0, 0, 0, 0,                       /* empty. */
0, 0, 0, 0, 0,
0, 0, 0, 0, 0,
0, 0, 0, 0, 0,
0, 0, 0, 0, 0,
0, 0, 0, 0, 0,
0, 0, 0, 0, 0,

5, 0, 0, 0, 0,                       /* wedges 6/15 nodes.  */
3, 1, 2, 3, 0, 
3, 4, 6, 5, 0, 
4, 1, 4, 5, 2,
4, 2, 5, 6, 3,
4, 3, 6, 4, 1, 
0, 0, 0, 0, 0, 

6, 0, 0, 0, 0,                       /* Bricks 8/20 nodes. */
4, 1, 2, 3, 4,
4, 5, 8, 7, 6,
4, 1, 5, 6, 2,
4, 2, 6, 7, 3,
4, 3, 7, 8, 4,
4, 4, 8, 5, 1,

4, 0, 0, 0, 0,                       /* tetrahedra, 4/10 nodes. */
3, 1, 2, 3, 0,
3, 1, 4, 2, 0, 
3, 2, 4, 3, 0, 
3, 3, 4, 1, 0, 
0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 

4, 0, 0, 0, 0,                       /* quadrilaterals, 4/8 nodes. */
2, 1, 2, 0, 0, 
2, 2, 3, 0, 0, 
2, 3, 4, 0, 0, 
2, 4, 1, 0, 0, 
0, 0, 0, 0, 0, 
0, 0, 0, 0, 0,

3, 0, 0, 0, 0,                       /* triangles, 3/6 nodes. */
2, 1, 2, 0, 0, 
2, 2, 3, 0, 0,
2, 3, 1, 0, 0, 
0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 
0, 0, 0, 0, 0                                                             };

int maxdimen, mindimen;    /* min and max dimensions found in model. */

#ifdef PROOL
#define BUFSIZE 512
#define TMPNAME "prool.tmp"
int max_nument;
char tmp_groupname[BUFSIZE];

int get_groupname(char *input, char *output)
// input - input line
// output - groupname
// return 0 if error, non zero if not error
{
char *cc;
output[0]=0;
strcpy(output,input);
cc=strchr(output,'\r');
if (cc) *cc=0;
cc=strchr(output,'\n');
if (cc) *cc=0;
cc=strchr(output,' ');
if (cc) *cc='_';
//printf("prool: get_groupname '%s'\n", output);
return strlen(output);
}

void process_e(char *filename)
{FILE *f1, *f2;
char buf[BUFSIZE], *pp;
char tmp_buf[BUFSIZE];
int i,len;

f1=fopen(filename,"r");
if (f1==NULL) {printf("prool_process_e: can't open input file `%s'\n",filename); return;}
f2=fopen(TMPNAME,"w");
if (f2==NULL) {printf("prool_process_e: can't open tmp file `%s'\n",TMPNAME); return;}

while(!feof(f1))
    {
    buf[0]=0;
    fgets(buf,BUFSIZE,f1);
#define LONG_NAME "PERMANENT GROUP"
    // replace LONG_NAME to "gr"
    if (memcmp(buf,LONG_NAME,strlen(LONG_NAME))==0)
	{
	strcpy(tmp_buf,buf);
	strcpy(buf,"gr");
	strcat(buf,pp+strlen(LONG_NAME));
	}
    // replace D to E
    len=strlen(buf);
    for (i=0;i<len;i++)
	{
	if (buf[i]=='D' && (i < (len-1) && (buf[i+1] == '+' || buf[i+1] == '-')))   buf[i]='E';
  }
    fputs(buf,f2);
    }
    
fclose(f1);
if (fclose(f2))
 {printf("prool_process_e: can't close tmp file `%s'\n",TMPNAME); return;}

}

void help(void)
{
printf
("unical1 prool mod\n\nusage: unical1 inputfile outputfile\nor\nunical1 -f inputfile outputfile\n\nProol: http://prool.kharkov.org\n\n");
}
#endif

int name_is_elem(char * groupname) // by prool
{
char prool_buffer[BUFSIZE];
int i, ret;
char *cc, *groupname0;

groupname0=groupname;

for(i=0;i<BUFSIZE;i++) prool_buffer[i]=0;

cc=prool_buffer;
while (*groupname)
	{
	*cc++=toupper(*groupname++);
	}
if(strstr(prool_buffer,"ELEM")==NULL) ret=0;
else ret=1;

//printf("prool debug name_is_elem() name='%s' upper='%s' ret=%i\n", groupname0, prool_buffer, ret);

return ret;
}

/*============================================================================*/
int main (int argc, char **argv)                                               {
/* printf("argc=%d\n", argc);  */
#ifdef PROOL
max_nument=0;
if(argc==3)
{
exitif(strlen(argv[1])>195, "unical: sorry input file name too long.", NULL);
strcpy(inname, argv[1]);
exitif(strlen(argv[1])>195, "unical: sorry output file name too long.", NULL);
strcpy(outname, argv[2]);
printf("unical1, Copyright(C) 2011 Bernhardi \n");
printf("some modif. by prool, 2015-2017. http://prool.kharkov.org\n");
printf("unical comes with ABSOLUTELY NO WARRANTY. This is free\n");
printf("software, and you are welcome to redistribute it under\n");
printf("certain conditions, see http://www.gnu.org/licenses/gpl.html\n\n");
full=0;
}
else
if(argc==4)
{
if (strcmp(argv[1],"-f"))
		{
		help();
		return 1;
		}
full=1;
strcpy(inname,argv[2]);
strcpy(outname,argv[3]);
}
else
{
help();
//printf("unical: usage: unical inputfile outputfile\n");
exit(2);
}
#else
if(argc==2)                                                               {
exitif(strlen(argv[1])>195, "unical: sorry file name too long.", NULL);
strcpy(problem, argv[1]);
full=0;                                                                   }
else if(argc==3)                                                          {
exitif(strncmp(argv[1],"-f",2)!=0, "unical: command not understood.", NULL);
exitif(strlen(argv[2])>195, "unical: sorry file name too long.", NULL);
strcpy(problem, argv[2]);
full=1;                                                                   }
else                                                                      {
printf("unical: usage: unical [-f] unv_filename (without extension .unv)\n");
exit(0);                                                                  }

printf("unical1, Copyright(C) 2011 Bernhardi \n");   
printf("unical comes with ABSOLUTELY NO WARRANTY. This is free\n");
printf("software, and you are welcome to redistribute it under\n");
printf("certain conditions, see http://www.gnu.org/licenses/gpl.html\n\n");

strcpy(inname, problem);
strcat(inname, ".unv");
#endif

#ifdef PROOL
process_e(inname); // convert exponent letter D+00 to E+00
strcpy(inname,TMPNAME);
#endif 

//printf("unical: Reading from file %s\n",inname);
fid = fopen(inname,"r");
exitif(fid==0,"Error reading from file ",inname);

countuni();     /* determine number of elements, nodes, etc. */
printf("unical: maxnode=%d, sumele=%d\n", maxnode, sumele);
allocate();     /* allocate large arrays. */
rewind(fid);
readnodes();    /* read nodes: coordinates. */
rewind(fid);
readelements(); /* read element connectivities. */
rewind(fid);
readgroups();   /* read group information. */

/* close universal file  */
//printf("unical: closing universal file %s\n", inname);
ret = fclose(fid);
exitif(ret < 0,"while attempting to close file ",inname);  

#ifndef PROOL
strcpy(outname, problem);
strcat(outname, ".inp");
#endif
printf("unical: writing to file %s\n",outname);
fid = fopen(outname,"w");
exitif(fid==0,"Error opening file ",outname);
writemesh(); /* write node point and element data to abaqus/calculix input file. */
surface();   /* process groups for boundary conditions. */

#ifndef PROOL
/* write standard ABAQUS procedure calls: *frequency step. */
fprintf(fid,"*material, name=steel\n");
fprintf(fid,"*elastic\n");
fprintf(fid,"209000.0, 0.3\n");
fprintf(fid,"*density\n");
fprintf(fid,"7.85e-09\n");
fprintf(fid,"*step\n");
fprintf(fid,"*frequency\n");
fprintf(fid,"12\n");
fprintf(fid,"*nodefile\n");
fprintf(fid,"u\n");
fprintf(fid,"*elfile, output=3D\n");
fprintf(fid,"s, e\n");
fprintf(fid,"*endstep\n");
#endif

/* close abaqus/calculix input file  */
//printf("unical: closing file %s\n", outname);
fprintf(fid,"\n"); // prool: end of file
ret = fclose(fid);
exitif(ret < 0,"while attempting to close file ",outname);  

printf("unical: conversion is done. check results carefully! \n");
printf("unical: returning to system. \n");
#ifdef PROOL
if (remove(TMPNAME)) {printf("can't delete tmp file `%s'\n",TMPNAME); exit(1);}
#endif
exit(0);                                                                      }
/*===========================================================================*/
void countuni()                                                               {
int i;
int ncard; 
int prool_i, prool_counter;

/* Start counting elements and nodes on data set. */
for(i=1;i<=15;i++) maxele[i]=0;
sumele=0;
maxnode=0;
maxgroup=0;
for(;;)                                                                  {

sret=fgets(line, maxline, fid);
if(sret==0)                                                           {
//printf("countunv: end-of-file while reading from %s\n", inname);
break;                                                                }
//printf("countunv: line=%s\n", line);

if(strncmp(line, "    -1",6)==0)                                      {
sret=fgets(line, maxline, fid);
if(sret==0)                                                       {
//printf("countunv: end-of-file while reading from %s\n", inname);
break;                                                            }
ret=sscanf(line, "%d", &key);
exitif(ret==0, "countunv: unexpected error 1  while reading from ", inname);
//printf("countunv: inname=%s, key=%d\n", inname, key);

if(key==2411)                                                     {
printf("countunv: processing dataset 2411, nodes.\n");
for(;;)                                                      {
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading Set 2411 from ", 
inname);
ret=sscanf(line, "%d", &key);
exitif(ret==0, "countunv: unexpected error 1  while reading Set 2411 from ", 
inname);
//printf("countunv:  node point number =%d\n", key);
if(key==-1) break;
maxnode++;
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading Set 2411 from ",
inname);                                                     }    }

else if(key==2412)                                                {
printf("countunv: processing dataset 2412, elements.\n");
for(;;)                                                      {
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2412 from ", 
inname);
//printf("countunv: line=%s\n", line);
ret=sscanf(line, "%d %d ", &key,&utype);
exitif(ret==0, "countunv: unexpected error 1 while reading set 2412 from ", 
inname);
if(key==-1) break;
//printf("countunv: key=%d, utype=%d\n", key, utype);
exitif(utype<1||utype>232, "countunv: wrong type of element.", 0);
for(i=1;i<=86;i++)                                     {
if(utype==eltypes[i][1])                          {      
eltype=eltypes[i][0];
btype=eltypes[i][2];                
break;                                            }    }
//printf("countunv: key=%d, utype=%d, eltype=%d\n", key,utype,eltype);
sumele++;
maxele[eltype]++;
//printf("maxele[%d]=%d\n",eltype,maxele[eltype]);

if(btype==1)                                           {
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2412 from ",
inname);                                               }

ncard=numnodes[eltype]/8;
if(numnodes[eltype]%8>0) ncard=ncard+1;
for(j=0;j<ncard;j++)                                   {
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2412 from ",
inname);                                               }    }     }

else if(key==2477)                                                {
printf("countunv: processing dataset 2477, groups.\n");
for(;;)                                                     {
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2477 from ", 
inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", &groupnumber,&dummy,
&dummy,&dummy,&dummy, &dummy,&dummy,&nument);
exitif(ret==0, "countunv: unexpected error 1 while reading set 2477 from ", 
inname);
printf("countunv: groupnumber=%d, nument=%d\n", groupnumber, nument);
if(groupnumber==-1) break;
maxgroup++;
exitif(nument<=0, "countunv: error #1 wrong nument <0", NULL);
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2477 from ",
inname);
ret=get_groupname(line,groupname);
exitif(ret==0, "countunv: error reading groupname.", NULL);
printf("countunv: groupname=%s\n", groupname);
for(i=1;i<=nument/2+nument%2;i++)                      {
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2477 from ",
inname);                                               }    }     }

else if(key==2477) { // prool: begin of 2477 first processing
printf("prool debug countunv: processing dataset 2477, groups.\n");
for(;;)                                                     {
sret=fgets(line, maxline, fid);
//printf("prool header of group=%s\n", line);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2477 from ", inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", &groupnumber,&dummy, &dummy,&dummy,&dummy, &dummy,&dummy,&nument);
exitif(ret==0, "countunv: unexpected error 1 while reading set 2477 from ", inname);
printf("prool debug countunv: groupnumber=%d, nument=%d\n", groupnumber, nument);
if(groupnumber==-1) break;
maxgroup++;
exitif(nument<=0, "countunv: error #2 wrong nument <=0", NULL);

/* calculate maximum. prool */
if (nument>max_nument) max_nument=nument;

sret=fgets(line, maxline, fid);
//printf("prool groupname=%s\n", line);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2477 from ", inname);
ret=get_groupname(line,groupname);
exitif(ret==0, "countunv: error reading groupname.", NULL);
printf("prool debug countunv: groupname=%s\n", groupname);

// by prool:
prool_counter=0;
for(;;)                      {// begin for
sret=fgets(line, maxline, fid);
exitif(sret==0, "countunv: unexpected end-of-line while reading set 2477 from ", inname);

//printf("prool group line=%s\n", line);
prool_i=sscanf(line, "%d %d %d %d %d %d %d %d", &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy);
//printf("prool debug sscanf ret=%i\n", prool_i);
if (prool_i==4)
	{// prool: short line
	prool_counter++;
	}
else
	prool_counter+=2;
if (prool_counter==nument) break;
}/*end for*/
}     } // prool: end of 2477 first processing

else                                                              {
for(;;)                                                     {
sret=fgets(line, maxline, fid);
if(sret==0)                                            {
//printf("countunv: end-of-file while reading from %s\n", inname);
break;                                                 }
if(strncmp(line, "    -1",6)==0) break;                     }     }
                                                                      }   }

printf("countunv: %d nodes found on dataset. \n", maxnode);
mindimen=10;
maxdimen=0;
for(i=1;i<=15;i++)                                                        {
if(maxele[i]>0)                                                       {
if(mindimen>dimensions[i])mindimen=dimensions[i];
if(maxdimen<dimensions[i])maxdimen=dimensions[i];
printf("countunv: %d elements of type %d: %s. \n", maxele[i],i,nomabq[i]);
                                                                      }   }
printf("countunv: %d groups found on .unv dataset. \n",maxgroup);
exitif(maxnode<=0, "countunv: no nodes found. Cannot proceed. ", 0);
exitif(sumele<=0, "countunv: no elements found. Cannot proceed. ", 0); 
printf("countunv: mindimen=%d, maxdimen=%d\n", mindimen, maxdimen);

numvol=0;
for(i=0;i<=15;i++) if(dimensions[i]==maxdimen) numvol+=maxele[i];
printf("countunv: %d elements of max dimension found.\n", numvol);
                                                                              }
/*===========================================================================*/
void allocate()                                                               {
int i;
/* allocate memory: Mesh. */
printf("allocate: allocating large data arrays. \n");
nodenumbers=(int*)malloc(sizeof(int)*maxnode);
exitif(nodenumbers==0,"allocate: allocation failure: node numbers.",NULL);
coords=(double*)malloc(sizeof(double)*3*maxnode);
exitif(coords==0,"allocate: allocation failure: node coordinates.",NULL);
groupnumbers=(int*)malloc(sizeof(int)*maxgroup);
exitif(groupnumbers==0,"allocate: allocation failure: group numbers.",NULL);
groupents=(int*)malloc(sizeof(int)*maxgroup);
exitif(groupents==0,"allocate: allocation failure: group lengths.",NULL);
groupnames=(char**)malloc(sizeof(char*)*maxgroup);
exitif(groupnames==0,"allocate: allocation failure: group names.",NULL);
groupmembers=(int**)malloc(sizeof(int*)*maxgroup);
exitif(groupmembers==0,"allocate: allocation failure: group members.",NULL);
grouptypes=(int**)malloc(sizeof(int*)*maxgroup);
exitif(grouptypes==0,"allocate: allocation failure: group types.",NULL);
grouppointers=(int**)malloc(sizeof(int*)*maxgroup);
exitif(grouppointers==0,"allocate: allocation failure: group pointers.",NULL);
groupeltypes=(int**)malloc(sizeof(int*)*maxgroup);
exitif(groupeltypes==0,"allocate: allocation failure: group eltypes.",NULL);

for(i=1;i<=12;i++)                                                       {
elem[i]=0;
if(maxele[i]>=0)                                                     {
elem[i]=(int*)malloc(sizeof(int)*numnodes[i]*(maxele[i]));
exitif(elem[i]==0,"allocate: allocation failure: elements.",NULL);
elnumbers[i]=(int*)malloc(sizeof(int)*(maxele[i]));
exitif(elnumbers[i]==0,"alloc failure: element numbers.",NULL);      }    }

surfnum=(int*)malloc(sizeof(int)*6*numvol);
exitif(surfnum==0,"allocate: allocation failure: surface element numbers.",NULL);
volelnum=(int*)malloc(sizeof(int)*6*numvol);
exitif(volelnum==0,"allocate: allocation failure: volume element numbers.",NULL);
facenum=(int*)malloc(sizeof(int)*6*numvol);
exitif(facenum==0,"allocate: allocation failure: face numbers.",NULL);
group=(int*)malloc(sizeof(int)*6*numvol);
exitif(group==0,"allocate: allocation failure: group numbers .",NULL);
                                                                              }
/*===========================================================================*/
void readnodes()                                                              {
int i;

numnode=0;
/* start reading nodes. */
for(;;)                                                                  {

sret=fgets(line, maxline, fid);
if(sret==0)                                                           {
//printf("readnodes: end-of-file while reading from %s\n", inname);
break;                                                                }

if(strncmp(line, "    -1",6)==0)                                      {
sret=fgets(line, maxline, fid);
if(sret==0)                                                       {
//printf("readnodes: end-of-file while reading from %s\n", inname);
break;                                                            }
ret=sscanf(line, "%d", &key);
exitif(ret==0, "readnodes: unexpected error 1  while reading from ", inname);
//printf("readnodes: inname=%s, key=%d\n", inname, key);

if(key==2411)                                                     {
for(;;)                                                      {
sret=fgets(line, maxline, fid);
exitif(sret==0, "readnodes: unexpected end-of-line reading Set 2411 from ", 
inname);
ret=sscanf(line, "%d", &key);
exitif(ret==0, "readnodes: unexpected error 1  while reading Set 2411 from ", 
inname);
//printf("readnodes:  node point number =%d\n", key);
if(key==-1) break;
numnode++;

exitif(numnode>maxnode,"error too many nodes found.\n", NULL);
nodenumbers[numnode-1]=key;
sret=fgets(line, maxline, fid);
exitif(sret==0, "readnodes: unexpected end-of-line while reading Set 2411 from ",
inname);
ret=sscanf(line, "%lf %lf %lf", &x,&y,&z);
exitif(ret==0, "readnodes: unexpected error 1  while reading Set 2411 from ", 
inname);
//printf("node=%d, x=%g, y=%g, z=%g\n", key,x,y,z);
coords[(numnode-1)*3]=x;
coords[(numnode-1)*3+1]=y;
coords[(numnode-1)*3+2]=z;                                   }    }

else                                                              {
for(;;)                                                     {
sret=fgets(line, maxline, fid);
if(sret==0)                                            {
//printf("readnodes: end-of-file while reading from %s\n", inname);
break;                                                 }     
if(strncmp(line, "    -1",6)==0) break;                     }    }   }   }    }
/*===========================================================================*/
void readelements()                                                           {
int i,j,k;
int ncard, *ln;
numele=0;
for(i=1;i<=12;i++) elnumber[i]=0;

/* start reading elements. */
for(;;)                                                                  {

sret=fgets(line, maxline, fid);
if(sret==0)                                                           {
//printf("readelements: #1 end-of-file while reading from %s\n", inname);
break;                                                                }
if(strncmp(line, "    -1",6)==0)                                      {
sret=fgets(line, maxline, fid);
if(sret==0)                                                       {
//printf("readnodes: end-of-file while reading from %s\n", inname);
break;                                                            }

ret=sscanf(line, "%d", &key);
exitif(ret==0, "readelements: unexpected error 1  while reading from ", inname);
//printf("readelements: inname=%s, key=%d\n", inname, key);

if(key==2412)                                                     {
for(;;)                                                      {
sret=fgets(line, maxline, fid);
exitif(sret==0, "readelements: unexpected end-of-line, reading set 2412 from ", 
inname);
ret=sscanf(line, "%d %d ", &key,&utype);
exitif(ret==0, "readelements: unexpected error 1 while reading set 2412 from ", 
inname);
if(key==-1) break;
//printf("readelements: key=%d, universal element type=%d\n", key, utype);
exitif(utype<1||utype>232, "readelements: wrong type of element.", 0);
for(i=1;i<=86;i++)                                     {
if(utype==eltypes[i][1])                          {      
eltype=eltypes[i][0];
btype=eltypes[i][2];                
break;                                            }    }
//printf("readelements: key=%d, utype=%d, eltype=%d\n", key,utype,eltype);
//printf("readelements: numnodes[eltype]=%d\n", numnodes[eltype]);
numele++;
elnumber[eltype]++;
exitif(elnumber[eltype]>maxele[eltype], 
"Sorry error too many elements.", NULL);
elnumbers[eltype][elnumber[eltype]-1]=key;

if(btype==1)                                           {
sret=fgets(line, maxline, fid);
exitif(sret==0, "readelements: unexpected end-of-line, reading set 2412 from ",
inname);                                               }

ncard=numnodes[eltype]/8;
if(numnodes[eltype]%8>0) ncard=ncard+1;
//printf("readelements: eltype=%d, ncard=%d\n", eltype, ncard);
for(j=0;j<ncard;j++)                                   {
sret=fgets(line, maxline, fid);
exitif(sret==0, "readelements: unexpected end-of-line, reading set 2412 from ",
inname);
ln=elnode+j*8;
ret=sscanf(line, " %d %d %d %d %d %d %d %d ", 
ln,ln+1,ln+2,ln+3,ln+4,ln+5,ln+6,ln+7);
exitif(ret<=0, "readelements: unexpected error 1 while reading elnode from ", 
inname);
//printf("readelements: element=%d, nodes=%d %d %d %d %d %d %d %d\n", elnumbers[eltype][elnumber[eltype]-1], *ln,*(ln+1),*(ln+2),*(ln+3),*(ln+4),*(ln+5),*(ln+6),*(ln+7));
                                                       }
for(i=1;i<=numnodes[eltype];i++)                       {
k=mapunv[eltype][i];
elem[eltype][numnodes[eltype]*(elnumber[eltype]-1)+i-1]=elnode[k-1];
                                                       }    }     }

else                                                              {
for(;;)                                                     {
sret=fgets(line, maxline, fid);
if(sret==0)                                            {
//printf("readelements: #2 end-of-file while reading from %s\n", inname);
break;                                                 }   
if(strncmp(line, "    -1",6)==0) break;                     }     }   }   }  
//printf("prool trace #01\n");
}
/*===========================================================================*/
void readgroups()                                                             {
int i,l;
int prool_counter;

//printf("prool trace #02\n");
numgroup=-1;
/* start reading groups. */
for(;;)                                                                  {

//printf("prool trace #03\n");
sret=fgets(line, maxline, fid);
//printf("prool trace #04\n");
if(sret==0)                                                           {
//printf("readgroups: end-of-file while reading from %s\n", inname);
break;                                                                }

if(strncmp(line, "    -1",6)==0)                                      {
sret=fgets(line, maxline, fid);
//printf("prool trace #05\n");
if(sret==0)                                                       {
//printf("readgroups: end-of-file while reading from %s\n", inname);
break;                                                            }

ret=sscanf(line, "%d", &key);
//printf("prool trace #06\n");
exitif(ret==0, "readnodes: unexpected error 1  while reading from ", inname);
//printf("readgroups: inname=%s, key=%d\n", inname, key);
//printf("prool trace #07\n");

if(key==2477)                                                     {
for(;;)                                                       {
sret=fgets(line, maxline, fid);
//printf("prool trace #08\n");
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ", 
inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", &groupnumber,&dummy,
&dummy,&dummy,&dummy, &dummy,&dummy,&nument);
exitif(ret==0, "readgroups: unexpected error 1 while reading set 2477 from ", 
inname);
printf("readgroups: groupnumber=%d, nument=%d\n", groupnumber, nument);
if(groupnumber==-1) break;
numgroup++;
exitif(numgroup>maxgroup, "readgroups: error too many groups.",NULL);
exitif(nument<=0, "readgroups: error #3 wrong nument <0", NULL);
groupnumbers[numgroup]=groupnumber;
groupents[numgroup]=nument;

/* allocate space for groups. */
groupnames[numgroup]=(char*)malloc(80*sizeof(int));
exitif(groupnames[numgroup]==0, 
"readgroups: sorry cannot allocate storage for groupnames. ",NULL);
groupmembers[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(groupmembers[numgroup]==0, 
"readgroups: cannot alloc storage for groupmembers. ",NULL);
grouptypes[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(grouptypes[numgroup]==0, 
"readgroups: cannot alloc storage for grouptypes. ",NULL);
grouppointers[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(grouppointers[numgroup]==0, 
"readgroups: cannot alloc storage for grouppointers. ",NULL);
groupeltypes[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(groupeltypes[numgroup]==0, 
"readgroups: cannot alloc storage for groupeltypes. ",NULL);

/* read group data. */
sret=fgets(line, maxline, fid);
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ",
inname);
ret=get_groupname(line,groupname);
exitif(ret==0, "readgroups: error reading groupname.", NULL);
strcpy(groupnames[numgroup],groupname);
printf("readgroups: groupname=%s\n", groupnames[numgroup]);
for(i=0;i<nument/2;i++)                                {
sret=fgets(line, maxline, fid);
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ",
inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", elnode,elnode+1,elnode+2,elnode+3,
elnode+4,elnode+5,elnode+6,elnode+7);
exitif(*(elnode)<=0||*(elnode+1)<=0||*(elnode+4)<=0||*(elnode+5)<=0,
"readgroups: error wrong group ID numbers.", NULL);
grouptypes[numgroup][2*i]=*elnode;
groupmembers[numgroup][2*i]=*(elnode+1);
grouptypes[numgroup][2*i+1]=*(elnode+4);
groupmembers[numgroup][2*i+1]=*(elnode+5);             }
if(nument%2==1)                                        {
sret=fgets(line, maxline, fid);
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ",
inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", elnode,elnode+1,elnode+2,elnode+3,
elnode+4,elnode+5,elnode+6,elnode+7);
exitif(*(elnode)<=0||*(elnode+1)<=0,
"readgroups: error wrong group ID numbers.", NULL);
grouptypes[numgroup][nument-1]=*elnode;
groupmembers[numgroup][nument-1]=*(elnode+1);          }

//printf("readgroups: List of group #%d, name= %s\n", groupnumbers[numgroup],groupnames[numgroup]);
for(i=0;i<nument;i++)                                  {
//printf("type=%d, entity=%d\n", grouptypes[numgroup][i],groupmembers[numgroup][i]);
                                                       } 
/* find pointers to element data of elements of this group. */
printf("readgroups: #1 indexing entities of group %s\n", groupnames[numgroup]);
for(i=0;i<nument;i++)                                  {
if(i%10000==0&&i!=0)printf("readgroups: processing entity %d\n",i);
groupeltypes[numgroup][i]=0;
grouppointers[numgroup][i]=0;
for(k=1;k<=12;k++)                                {
for(l=0;l<elnumber[k];l++)                   {
if(groupmembers[numgroup][i]==elnumbers[k][l]
 &&grouptypes[numgroup][i]==8)          {
groupeltypes[numgroup][i]=k;
grouppointers[numgroup][i]=l;
goto exitloop;                          }    }    }
exitif(NULL, "readgroups: error: Element listed in group does not exist.", NULL);
exitloop:{}                                            }
                                                            }     }

else if(key==2477) { // prool: begin 2477 second processing
//printf("prool trace #09\n");
for(;;)                                                       {
sret=fgets(line, maxline, fid);
//printf("prool trace #0a\n");
//printf("prool second proc. group header '%s'\n", line);
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ", 
inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", &groupnumber,&dummy,
&dummy,&dummy,&dummy, &dummy,&dummy,&nument);
//printf("prool trace #0b\n");
exitif(ret==0, "readgroups: unexpected error 1 while reading set 2477 from ", 
inname);
printf("prool debug readgroups: groupnumber=%d, nument=%d\n", groupnumber, nument);
if(groupnumber==-1) break;
numgroup++;
exitif(numgroup>maxgroup, "readgroups: error too many groups.",NULL);
exitif(nument<=0, "readgroups: error #4 wrong nument <0", NULL);
groupnumbers[numgroup]=groupnumber;
groupents[numgroup]=nument;

/* allocate space for groups. */
groupnames[numgroup]=(char*)malloc(80*sizeof(int));
exitif(groupnames[numgroup]==0, 
"readgroups: sorry cannot allocate storage for groupnames. ",NULL);
groupmembers[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(groupmembers[numgroup]==0, 
"readgroups: cannot alloc storage for groupmembers. ",NULL);
grouptypes[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(grouptypes[numgroup]==0, 
"readgroups: cannot alloc storage for grouptypes. ",NULL);
grouppointers[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(grouppointers[numgroup]==0, 
"readgroups: cannot alloc storage for grouppointers. ",NULL);
groupeltypes[numgroup]=(int*)malloc(sizeof(int)*nument);
exitif(groupeltypes[numgroup]==0, 
"readgroups: cannot alloc storage for groupeltypes. ",NULL);

/* read group data. */
sret=fgets(line, maxline, fid);
//printf("prool trace #0c\n");
//printf("prool second proc. groupname '%s'\n", line);
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ",
inname);
ret=get_groupname(line,groupname);
exitif(ret==0, "readgroups: error reading groupname.", NULL);
strcpy(groupnames[numgroup],groupname);
//printf("prool trace #0d\n");
printf("prool debug readgroups: groupname=%s\n", groupnames[numgroup]);

prool_counter=0;
i=0;
for(;;) {
sret=fgets(line, maxline, fid);
//printf("prool trace #0e\n");
//printf("prool second proc. group line '%s'\n", line);
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ", inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", elnode,elnode+1,elnode+2,elnode+3, elnode+4,elnode+5,elnode+6,elnode+7);
//printf("prool trace #0f\n");

// by prool:
if (ret==4)
	{// prool: short line
//printf("prool trace #0g\n");
	exitif(*(elnode)<=0||*(elnode+1)<=0, "prool:: readgroups: error #2 wrong group ID numbers.", NULL);
	grouptypes[numgroup][i]=*elnode;
	groupmembers[numgroup][i]=*(elnode+1);
	prool_counter++;i++;
	}
else
	{
//printf("prool trace #0h\n");
	exitif(*(elnode)<=0||*(elnode+1)<=0||*(elnode+4)<=0||*(elnode+5)<=0, "prool:: readgroups: error wrong group ID numbers.", NULL);
	grouptypes[numgroup][i]=*elnode;
	groupmembers[numgroup][i]=*(elnode+1);
	grouptypes[numgroup][i+1]=*(elnode+4);
	groupmembers[numgroup][i+1]=*(elnode+5);
	prool_counter+=2;i+=2;
	}
//printf("prool trace #0i\n");
if (prool_counter>=nument) break;
//printf("prool trace #0j\n");
} // end for

//printf("prool trace #0k\n");
#if 0 // prool
if(nument%2==1)                                        {
sret=fgets(line, maxline, fid);
printf("prool second proc. group final line '%s'\n", line);
exitif(sret==0, "readgroups: unexpected end-of-line while reading set 2477 from ", inname);
ret=sscanf(line, "%d %d %d %d %d %d %d %d", elnode,elnode+1,elnode+2,elnode+3, elnode+4,elnode+5,elnode+6,elnode+7);
exitif(*(elnode)<=0||*(elnode+1)<=0, "readgroups: error wrong group ID numbers.", NULL);
grouptypes[numgroup][nument-1]=*elnode;
groupmembers[numgroup][nument-1]=*(elnode+1);          } // end if
#endif

//printf("prool debug readgroups: List of group #%d, name= %s\n", groupnumbers[numgroup],groupnames[numgroup]);
for(i=0;i<nument;i++)                                  {
//printf("prool debug type=%d, entity=%d\n", grouptypes[numgroup][i],groupmembers[numgroup][i]);
                                                       }
/* find pointers to element data of elements of this group. */
//printf("prool debug readgroups: #2 indexing entities of group %s\n", groupnames[numgroup]);
for(i=0;i<nument;i++)                                  {
if(i%10000==0&&i!=0)printf("prool debug readgroups: processing entity %d\n",i);
groupeltypes[numgroup][i]=0;
grouppointers[numgroup][i]=0;
for(k=1;k<=12;k++)                                {
for(l=0;l<elnumber[k];l++) {// for l begin
if (groupmembers[numgroup][i]==elnumbers[k][l]) // prool: i don't know about this expression
	{
	//printf("prool's debug: max_nument=%i\n", max_nument);
	if ((groupents[numgroup]==max_nument) || name_is_elem(groupnames[numgroup]))
		{
		if (grouptypes[numgroup][i]==8) { groupeltypes[numgroup][i]=k; grouppointers[numgroup][i]=l; goto exitloop2; }
		}
	else if (grouptypes[numgroup][i]==7) {groupeltypes[numgroup][i]=k; grouppointers[numgroup][i]=l; goto exitloop2; }
	}
} // for l end
}
exitif(NULL, "readgroups: error: Element listed in group does not exist.", NULL);
exitloop2:{}                                            }
                                                            }     } // prool: end of 2477 second processing

else                                                              {
for(;;)                                                     {
sret=fgets(line, maxline, fid);
if(sret==0)                                            {
//printf("readgroups: end-of-file while reading from %s\n", inname);
break;                                                 }     
if(strncmp(line, "    -1",6)==0) break;                     }    }   }   }    }
/*============================================================================*/
void writemesh()                                                              {
int i,j,k,l,m,n,dim;

printf("writemesh:  writing node point data. \n");
#ifdef PROOL
fprintf(fid, "*node, nset=Nall");
#else
fprintf(fid, "*node, nset=nall");
#endif
for(i=0;i<maxnode;i++)                                                   {
if(i%10000==0) printf("writemesh: writing node %d\n", i);
#ifdef PROOL
fprintf(fid, "\n%d, %e, %e, %e",
#else
fprintf(fid, "\n%d, %g, %g, %g", 
#endif
nodenumbers[i], coords[3*i],coords[3*i+1],coords[3*i+2]);                }

printf("writemesh: writing groups of nodes: *NSETs. \n");
for(i=0;i<maxgroup;i++)                                                  {
nument=groupents[i];
j=0;
for(k=0;k<nument;k++)                                               {
if(grouptypes[i][k]==7)                                        {
if(j==0) fprintf(fid,"\n*nset, nset=%s\n", groupnames[i]);
j++;
if(j%8==0)fprintf(fid,"\n");
fprintf(fid,"%d, ",groupmembers[i][k]);                        }    }    }
fprintf(fid,"\n");

printf("writemesh: writing element connectivity data. \n");
k=0;
for(i=1;i<=12;i++)                                                       {
if(elnumber[i]>0&&(dimensions[i]==maxdimen||full==1))          {
printf("writemesh: processing elset %s\n", nomabq[i]);
printf("writemesh: i=%d, dimensions[i]=%d, maxdimen=%d\n", 
i, dimensions[i], maxdimen);
/*if(dimensions[i]==1)                                           { 
fprintf(fid, "*beam section, section=rect, material=steel, elset=%s\n", 
nomabq[i]);
fprintf(fid, "1.0, 1.0\n");                                    }
if(dimensions[i]==2)                                           {
fprintf(fid, "*shell section, material=steel, elset=%s\n", nomabq[i]);
fprintf(fid, "1.0\n");
                                                               }
if(dimensions[i]==3)                                           {
fprintf(fid, "*solid section, material=steel, elset=%s\n", nomabq[i]);
                                                               } */
fprintf(fid, "*element, elset=%s,type=%s\n", nomabq[i], nomabq[i]);
for(l=0;l<elnumber[i];l++)                                     {
k++;
if(k%10000==0)printf("writemesh: writing element %d\n",k);
fprintf(fid, "%d, ",elnumbers[i][l]);
for(n=0;n<numnodes[i];n++)                                {
fprintf(fid, "%d, ",elem[i][numnodes[i]*l+n]);
if(n==12) fprintf(fid, "\n");                             }
fprintf(fid, "\n");                                            }     }    }

printf("writemesh: writing groups of elements: *ELSETs. \n");
for(i=0;i<maxgroup;i++)                                                   {
printf("writemesh: processing group %s\n", groupnames[i]);
nument=groupents[i];
j=0;
for(k=0;k<nument;k++)                                               {
if(dimensions[groupeltypes[i][k]]==maxdimen||full==1)          {
if(j==0) fprintf(fid,"\n*elset, elset=%s\n", groupnames[i]);
j++;
if(j%8==0)fprintf(fid,"\n");
fprintf(fid,"%d, ",groupmembers[i][k]);                        }    }    }
                                                                               }
/*==============================================================================*/
void surface()                                                                 {
int i,j,k,l,m,n,dim;
int o,p,q,r,s;
int facnod, nodcount;
int facepoint; 
int maxface;
int surfcounter;

printf("surface:  generating surfaces. \n");
j=0;
surfcounter=-1;

/* all element types: */
for(i=1;i<=15;i++)                                                        {

/* surfaces are checked for maxdimen elements only. */
if(dimensions[i]==maxdimen&&elnumber[i]>0)                              {

/* facepoint points to faces table. */
facepoint=elpointer[i];

/* determine number of faces at this (volume or plane) element type. */
maxface=faces[facepoint][0][0];

/* start loop on all surface element groups. */
for(n=0;n<maxgroup;n++)                                              {

/* loop on all elements within this group. */
nument=groupents[n];
for(k=0;k<nument;k++)                                             {

/* elements only (no nodes) */
if(grouptypes[n][k]==8)                                        {

/* check dimension of and pointers (l,m) to element of this group. */
l=groupeltypes[n][k];
dim=dimensions[l];
m=grouppointers[n][k];

//printf("surface: vol type i=%d, surf type l=%d, surf number m=%d, dim=%d \n", i,l,m,dim);

/* if dimension fits, e.g. .eq. maxdimen-1, search for matching surface. */
if(dim==maxdimen-1  && (l==smatch[0][i]||l==smatch[1][i]))  {
j++;
if(j%1000==0) printf("surface: converting surface element %d\n", k+1);

/* loop on volume elements */
for(o=0;o<elnumber[i];o++)                              { 

//printf("surf=%d, vol=%d, facepoint=%d, maxface=%d\n", m,o,facepoint, maxface);

/* start loop on all surfaces of vol element */
for(p=1;p<=maxface;p++)                             { 

/* number of nodes at this surface. */
facnod=faces[facepoint][p][0];
//printf("facnod=%d\n", facnod);

/* loop on all edge nodes on surface of volume element*/
nodcount=0;
for(q=1;q<=facnod;q++)                           {
s=faces[facepoint][p][q];

/* loop on all edge nodes of surface element*/
for(r=1;r<=facnod;r++)                        { 
if(elem[i][numnodes[i]*o+s-1]==elem[l][numnodes[l]*m+r-1])nodcount++; 
                                              }  
if(nodcount<q) break;                            }
//printf("n=%d, o=%d, p=%d, nodcount=%d\n", n,o,p,nodcount);
if(facnod==nodcount)                             {
surfcounter++;
if(surfcounter>6*numvol)                     {
printf("surface: error too many surfaces found.\n");
printf("surface: surfcounter=%d\n", surfcounter);
exit(0);                                     }
//printf("facnod=%d, nodcount=%d\n", facnod, nodcount);
//printf("surfcounter=%d\n", surfcounter);
surfnum[surfcounter]=elnumbers[l][m];
volelnum[surfcounter]=elnumbers[i][o];
facenum[surfcounter]=p;
group[surfcounter]=n;
goto endloopk;                                   }
/* end loop p, on all surfaces of vol element */      }
/* end loop o, all volume elements. */                   } 
/* end if, dim==maxdimen-1 */                               } 
/* end if on grouptype==7. */                                  }
endloopk:{}
/* end loop k, all elements within surface element group. */      }  
/* end loop n, on all surface element groups. */                     }
/* end if, dimensions==maxdimen. */                                     }
/* end loop, all element types. */                                         }

if(surfcounter<=0)                                                         {
printf("surface: no surfaces found.\n");
return;                                                                    }
printf("surface: surfcounter=%d surfaces found.\n",surfcounter);
/* if(surfcounter>0)                                                          {
printf("surface: list of surface elements. \n");
printf("    n       surfnum  volelnum  facenum    group\n");
for(n=0;n<=surfcounter;n++) printf("%5d %10d %10d %10d %10d\n",
n,surfnum[n],volelnum[n],facenum[n],group[n]);                             }  */

/* ecrire les element surface sets correspondant aux groupes. */

/* This is "F-groups". Disable this or not - is question! Prool. */
for(m=0;m<maxgroup;m++)                                                    {
printf("surface: processing group set %d, %s\n", m, groupnames[m]);
for(p=1;p<=6;p++)                                                      {
n=0;
for(q=0;q<=surfcounter;q++)                                         {
if(group[q]==m&&facenum[q]==p)                                   {
n++;
if(n%8==0) fprintf(fid,"\n");
if(n==1) fprintf(fid,"\n*elset, elset=%sF%d\n", groupnames[m],p);
fprintf(fid,"%d, ",volelnum[q]);                                 }   }  }  }
fprintf(fid, "\n");

/* generate ABAQUS input lines for element surfaces. */
for(m=0;m<maxgroup;m++)                                                    {
printf("surface: processing group set %d, %s\n ", m, groupnames[m]);
i=0;
for(p=1;p<=6;p++)                                                      {
j=0;
for(q=0;q<=surfcounter;q++)                                         {
if(group[q]==m&&facenum[q]==p)                                   {
i++;j++;
if(i==1) fprintf(fid,"*surface, type=element, name=s%s\n", groupnames[m]);
if(j==1) fprintf(fid,"%sF%d, S%d\n", groupnames[m],p,p);         }   }  }  }
fprintf(fid, "\n");                                                            }
/*==============================================================================*/
void exitif(int expression,char* message, char* arg)                           {
if(expression==0) return;
if(message!=0) fprintf(stdout, "%s", message);
if(arg!=0) fprintf(stdout, "%s", arg);
fprintf(stdout, "\nexiting.\n");
#ifdef PROOL
if (remove(TMPNAME)) {printf("exitif(): can't delete tmp file `%s'\n",TMPNAME);}
#endif
exit(0);                                                                       }
/*==============================================================================*/

